﻿/*  Author:     Ben Hall
 *  File:       GameCharacter.cs
 *  Date:       4/28/2011
 *  
 *  Purpose:    GameCharacter.cs is a class that is extended by the AICharacter and PlayerCharacter
 *              classes.  It is used to represent any characters in the game and contains methods
 *              used by both classes, like determining if a skill is known by the character, or if
 *              the character is affected by any status effects.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Media.Imaging;

namespace OnlineGame
{
    public struct CharStat
    {
        public int stat;
        public const int MIN_VALUE = 1;
        public const int MAX_VALUE = 100;

        public CharStat(int statIn)
        {
            if (statIn < MIN_VALUE)
            {
                statIn = MIN_VALUE;
            }
            else if (statIn > MAX_VALUE)
            {
                statIn = MAX_VALUE;
            }
            stat = statIn;
        }
    }

    public abstract class GameCharacter
    {
        protected String characterName;
        protected BitmapImage portrait;
        protected BitmapImage characterIcon;

        protected CharStat strength;
        protected CharStat stamina;
        protected CharStat agility;
        protected CharStat intelligence;
        protected CharStat faith;
        protected CharStat persona;

        protected int currentHealth;
        protected int maxHealth;
        protected int currentMagic;
        protected int maxMagic;
        protected int level;

        protected int minDamage;
        protected int maxDamage;
        protected double armor;

        //STATUS EFFECTS//
        protected bool ranAway;
        protected bool defending;
        protected bool poisoned;
        protected bool covering;
        //bool sleeping;
        protected GameCharacter coveredBy;
        protected GameCharacter characterCovered;

        protected float pDefenseChange;
        protected float mDefenseChange;

        //SKILLS//
        protected bool[] skillsKnown;


        public GameCharacter()
        {
        }

        public GameCharacter(String name, int levelIn, int strIn, int stmIn, int aglIn, int intIn, int fthIn, int prsIn)
        {
            this.characterName = name;
            this.strength = new CharStat(strIn);
            this.stamina = new CharStat(stmIn);
            this.agility = new CharStat(aglIn);
            this.intelligence = new CharStat(intIn);
            this.faith = new CharStat(fthIn);
            this.persona = new CharStat(prsIn);
            this.level = levelIn;

            this.maxHealth = DetermineMaxHealth();
            this.currentHealth = maxHealth;
            this.maxMagic = DetermineMaxMagic();
            this.currentMagic = maxMagic;

            this.defending = false;
            this.poisoned = false;
            this.covering = false;
            this.coveredBy = null;
            this.characterCovered = null;
            this.pDefenseChange = 0;
            this.mDefenseChange = 0;

            this.skillsKnown = new bool[MainPage.skillList.Length];
            for (int i = 0; i < skillsKnown.Length; i++)
            {
                skillsKnown[i] = false;
            }

        }

        public BitmapImage GetPortrait()
        {
            return this.portrait;
        }
        public void SetPortrait(BitmapImage portraitIn)
        {
            this.portrait = portraitIn;
        }
        public BitmapImage GetIcon()
        {
            return this.characterIcon;
        }
        public void SetIcon(BitmapImage iconIn)
        {
            this.characterIcon = iconIn;
        }
        
        /*  DetermineMaxHealth() determines the GameCharacter's maximum health based on its stamina stat,
         *  then returns that value.
         *  Base value is twice of the character's stamina.
         *  Bonus value is equal to 10% of the character's base health per character level
         */
        protected int DetermineMaxHealth()
        {
            int baseHealth = (2 * this.stamina.stat);
            float bonusHealth = (2 * this.stamina.stat) * (0.10f * this.level);
            int health = baseHealth + (int)bonusHealth;
            return health;
        }

        /*  DetermineMaxMagic() determines the GameCharacter's maximum magic based on its intelligence
         *  stat, then returns that value.
         *  Base value is 150% of the character's intelligence.
         *  Bonus value is equal to 10% of the character's base intelligence per character level
         */
        protected int DetermineMaxMagic()
        {
            float baseMagic = (1.5f * this.intelligence.stat);
            float bonusMagic = (1.5f * this.intelligence.stat) * (0.10f * this.level);
            int magic = (int)baseMagic + (int)bonusMagic;
            return magic;
        }

        public double GetArmor()
        {
            return this.armor;
        }

        public void SetArmor(double armorIn)
        {
            this.armor = armorIn;
        }

        public int GetMinDamage()
        {
            return this.minDamage;
        }

        public int GetMaxDamage()
        {
            return this.maxDamage;
        }

        public void SetDamage(int minDamageIn, int maxDamageIn)
        {
            this.minDamage = minDamageIn;
            if (maxDamageIn <= minDamageIn)
            {
                this.maxDamage = minDamageIn;
            }
            else
            {
                this.maxDamage = maxDamageIn;
            }
        }

        public String GetName()
        {
            return this.characterName;
        }
        
        public int[] GetStats()
        {
            int[] stats = new int[6];
            stats[0] = strength.stat;
            stats[1] = stamina.stat;
            stats[2] = agility.stat;
            stats[3] = intelligence.stat;
            stats[4] = faith.stat;
            stats[5] = persona.stat;
            return stats;
        }

        public int GetLevel()
        {
            return this.level;
        }

        public int GetMaxHealth()
        {
            return this.maxHealth;
        }

        public int GetCurrentHealth()
        {
            return this.currentHealth;
        }

        public void SetCurrentHealth(int healthIn)
        {
            this.currentHealth = healthIn;
        }

        /*  GetHealthPercent() returns a float value equal to the ratio of currentHealth/maxHealth
         */
        public float GetHealthPercent()
        {
            float percent = (float)this.currentHealth / (float)this.maxHealth;
            return percent;
        }
        
        public int GetMaxMagic()
        {
            return this.maxMagic;
        }

        public int GetCurrentMagic()
        {
            return this.currentMagic;
        }

        public void SetCurrentMagic(int magicIn)
        {
            this.currentMagic = magicIn;
        }

        /*  GetMagicPercent() returns a float value equal to the ratio of currentMagic/maxMagic
         */
        public float GetMagicPercent()
        {
            float percent = (float)this.currentMagic / (float)this.maxMagic;
            return percent;
        }

        public int GetStrength()
        {
            return this.strength.stat;
        }

        public int GetStamina()
        {
            return this.stamina.stat;
        }

        public int GetAgility()
        {
            return this.agility.stat;
        }

        public int GetIntelligence()
        {
            return this.intelligence.stat;
        }

        public int GetFaith()
        {
            return this.faith.stat;
        }

        public int GetPersona()
        {
            return this.persona.stat;
        }

        //STATUS SETTERS/GETTERS
        public bool IsDefending()
        {
            return defending;
        }
        public void SetDefending(bool defendStatus)
        {
            this.defending = defendStatus;
        }

        public bool IsPoisoned()
        {
            return poisoned;
        }
        public void SetPoisoned(bool poisonStatus)
        {
            this.poisoned = poisonStatus;
        }

        public bool IsCovering()
        {
            return this.covering;
        }
        public void SetCovering(bool coveringStatus)
        {
            this.covering = coveringStatus;
        }


        public bool IsCovered()
        {
            if (this.coveredBy != null)
            {
                return true;
            }
            else
            {
                return false;
            }
        }
        public GameCharacter GetCoveredBy()
        {
            return this.coveredBy;
        }
        public void SetCoveredBy(GameCharacter coveringCharacter)
        {
            this.coveredBy = coveringCharacter;
        }

        public bool IsCoveringCharacter()
        {
            if (this.characterCovered != null)
            {
                return true;
            }
            else
            {
                return false;
            }
        }
        public GameCharacter GetCharacterCovered()
        {
            return this.characterCovered;
        }
        public void SetCharacterCovered(GameCharacter coveredCharacter)
        {
            this.characterCovered = coveredCharacter;
        }

        public float GetPDefenseChange()
        {
            return this.pDefenseChange;
        }
        public void SetPDefenseChange(float changeIn)
        {
            this.pDefenseChange = changeIn;
        }

        public float GetMDefenseChange()
        {
            return this.mDefenseChange;
        }
        public void SetMDefenseChange(float changeIn)
        {
            this.mDefenseChange = changeIn;
        }

        //
        public bool[] GetSkillsKnown()
        {
            return this.skillsKnown;
        }

        public void RemoveSkill(int skillNum)
        {
            this.skillsKnown[skillNum] = false;
        }
        public void AddSkill(int skillNum)
        {
            this.skillsKnown[skillNum] = true;
        }
        public bool AddSkill(String skillNameIn)
        {
            for (int i = 0; i < MainPage.skillList.Length; i++)
            {
                if (MainPage.skillList[i] != null)
                {
                    if (MainPage.skillList[i].GetSkillName().Equals(skillNameIn))
                    {
                        AddSkill(i);
                        return true;
                    }
                }
            }

            return false;

        }
        public bool IsSkillKnown(int skillNum)
        {
            return this.skillsKnown[skillNum];
        }
        public bool IsSkillKnown(String skillNameIn)
        {
            return this.skillsKnown[GetSkillWithName(skillNameIn)];
        }

        /*  GetSkillWithName() returns the skill ID of the skill with the matching name
         * 
         *  Returns -1 if no skill with that name exists.
         */
        public int GetSkillWithName(String skillNameIn)
        {
            for (int i = 0; i < MainPage.skillList.Length; i++)
            {
                if (MainPage.skillList[i] != null)
                {
                    if (MainPage.skillList[i].GetSkillName().Equals(skillNameIn))
                    {
                        return i;
                    }
                }
            }
            return -1;
        }


        /*  CanUseSkill() takes in a skillID and checks if the character knows and can use the skill,
         *  then returns a boolean value based on this.
         *  
         *  Returns true only when the character knows the skill and has enough Magic to use it.
         */
        public bool CanUseSkill(int skillIn)
        {
            if (this.IsSkillKnown(skillIn) == false)
            {
                //return false if skill is not known
                return false;
            }
            else if (this.GetCurrentMagic() < MainPage.skillList[skillIn].GetSkillCost())
            {
                //return false if AICharacter does not have enough Magic to use the skill.
                return false;
            }
            else
            {
                return true;
            }
        }
    }
}
